<?php

namespace SilkPanel\Installer;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use SilkPanel\Installer\Checks\{
    Check,
    DatabaseCheck,
    RequiredTablesCheck,
    PhpExtensionsCheck,
    StoragePermissionsCheck,
};

class SetupManager
{
    private const CACHE_KEY = 'silkpanel_setup_status';

    private array $checks = [];

    public function __construct()
    {
        $this->registerDefaultChecks();
    }

    private function registerDefaultChecks(): void
    {
        $this->checks = [
            new PhpExtensionsCheck(),
            new DatabaseCheck(),
            new RequiredTablesCheck(),
            new StoragePermissionsCheck(),
        ];
    }

    public function getChecks(): array
    {
        return $this->checks;
    }

    public function runAllChecks(): array
    {
        $results = [];

        foreach ($this->checks as $check) {
            $results[$check->getName()] = [
                'description' => $check->getDescription(),
                'result' => $check->run(),
            ];
        }

        return $results;
    }

    public function isInstalled(): bool
    {
        return (bool) Cache::get(self::CACHE_KEY);
    }

    public function verifySetupKey(string $key): bool
    {
        $configKey = $key ?: config('silkpanel.api_key');

        if (! $configKey) {
            return false;
        }

        try {
            $response = Http::post('https://ddev-silkpanel-saas-web/api/verify-key', [
                'key' => $configKey,
            ]);

            if (! $response->ok()) {
                \Log::warning('Setup key verification failed: HTTP error', [
                    'key' => $configKey,
                    'status' => $response->status(),
                ]);
                return false;
            }

            $data = $response->json();
            $isValid = $data['valid'] ?? false;
            $client = $data['client'] ?? null;
            $validUntil = $data['valid_until'] ?? null;

            \Log::info('Setup key verification result', [
                'key' => $configKey,
                'valid' => $isValid,
                'client' => $client,
                'valid_until' => $validUntil,
            ]);

            return $isValid;
        } catch (\Exception $e) {
            \Log::error('Setup key verification exception', [
                'key' => $configKey,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    public function completeSetup(string $apiKey): bool
    {
        if (! $this->verifySetupKey($apiKey)) {
            return false;
        }

        foreach ($this->checks as $check) {
            $result = $check->run();
            if (! $result->passed) {
                return false;
            }
        }

        Cache::put(self::CACHE_KEY, true, now()->addYears(1));

        return true;
    }

    public function resetSetup(): void
    {
        Cache::forget(self::CACHE_KEY);
    }
}
