<?php

namespace SilkPanel\Installer\Commands;

use Illuminate\Console\Command;
use SilkPanel\Installer\SetupManager;

class SetupCommand extends Command
{
    protected $signature = 'silkpanel:setup';
    protected $description = 'Run the SilkPanel setup wizard';

    public function handle(SetupManager $setupManager): int
    {
        $this->info('╔════════════════════════════════════════╗');
        $this->info('║      SilkPanel Setup Wizard            ║');
        $this->info('╚════════════════════════════════════════╝');
        $this->newLine();

        // Check if already installed
        if ($setupManager->isInstalled()) {
            $this->warn('⚠️  SilkPanel is already installed!');
            if (! $this->confirm('Do you want to reset and reinstall?')) {
                return 0;
            }
            $setupManager->resetSetup();
        }

        // Step 1: Run checks
        $this->info('Step 1/2: Running system checks...');
        $this->newLine();

        $results = $setupManager->runAllChecks();
        $allPassed = true;

        foreach ($results as $checkName => $data) {
            $result = $data['result'];
            $icon = $result->passed ? '✅' : '❌';
            $status = $result->passed ? 'green' : 'red';

            $this->line("<fg={$status}>{$icon} {$checkName}</>");
            $this->line("   {$result->message}");

            if ($result->hint) {
                $this->line("<fg=yellow>   💡 {$result->hint}</>");
            }

            if (! $result->passed) {
                $allPassed = false;
            }
        }

        $this->newLine();

        if (! $allPassed) {
            $this->error('❌ Setup checks failed. Please fix the issues above and try again.');
            return 1;
        }

        $this->info('✅ All checks passed!');
        $this->newLine();

        $this->info('Setup wizard completed. All checks are passing.');
        $this->info('Use your api key to complete installation:');
        $this->newLine();

        $apiKey = $this->ask('Enter your key');

        if (! $apiKey) {
            $this->warn('⚠️  Key not provided. You can complete setup later.');
            return 0;
        }

        if (! $setupManager->verifySetupKey($apiKey)) {
            $this->error('❌ Invalid or expired setup key.');
            return 1;
        }

        if ($setupManager->completeSetup($apiKey)) {
            $this->info('✅ Setup completed successfully!');
            $this->info('SilkPanel CMS is now ready to use.');
            return 0;
        }

        $this->error('❌ Setup completion failed.');
        return 1;

        return 0;
    }
}
